{$LONGSTRINGS OFF}
unit Win2Tree;

interface
Uses
  Classes, Windows, SysUtils, Menus, Sticky, Main;

Type
  PWin2TreeElement =^TWin2TreeElement;
  TWin2TreeElement = record
    Owner  : PWin2TreeElement;  // Parent window
    Handle : THandle;           // Handle of this window
    Title  : ShortString;       // Caption
    WClass : ShortString;       // Class name
    Rect   : TRect;             // Size
    Childs : TList;             // List of sub windows (not exactly childs)
  end;

Function  CreateNewElement(Owner: PWin2TreeElement; Handle: THandle): PWin2TreeElement;
Function  SearchElement(Root: PWin2TreeElement; Handle: THandle): PWin2TreeElement;
Procedure DestroyTree(Root: PWin2TreeElement);
Function LoadTree: PWin2TreeElement;
Procedure BuildMenu(Root: PWin2TreeElement; PopupMenu:TMenuItem; OnClick:TNotifyEvent; ShowCordinates:Boolean; StickyWindows:TStickyWindows);
Procedure SortTree(Root: PWin2TreeElement);
Function CountWindowsByTitle(Const Caption : String):Integer;
Function MyIsWindowVisible(H:THandle):Boolean;
function  GetWindowTitle(W:THandle):String; 
function GetWindowClass(W:THandle):String;

implementation
Uses
  SafeMem, Forms;

Function MyIsWindowVisible(H:THandle):Boolean; 
Var
  WRect: TRect;
  WPos : TWindowPlacement;
  S    : ShortString;
Begin
  S[0]:=Char(GetWindowText(H, @S[1], 200));
  Result:=IsWindowVisible(H)or(fMain.FindHandleInHiddenList(H)>=0);
  If Result Then Begin
    GetWindowRect(H, WRect);
    GetWindowPlacement(H, @WPos);
    Result:=not((WRect.Left=WPos.ptMinPosition.x)and(WRect.Top=WPos.ptMinPosition.y));
  End;
End;

Function CreateNewElement(Owner: PWin2TreeElement; Handle: THandle): PWin2TreeElement; 
Var
  Element   : PWin2TreeElement;
  ClassName : ShortString;
Begin
  ClassName[0] := Char(GetClassName(Handle, @ClassName[1], SizeOf(ClassName)-1));

  SafeGetMem(Element, SizeOf(TWin2TreeElement));
  Element^.Handle:=Handle;
  Element^.Title[0]:=Char(GetWindowText(Handle, @Element^.Title[1], SizeOf(Element^.Title)-1));
  Element^.WClass:=ClassName;

  GetWindowRect(Handle, Element^.Rect);
  If Owner<>nil Then Begin
    If Owner.Childs=nil Then Owner.Childs:=TList.Create;
    Owner^.Childs.Add(Element);
  End;
  Result:=Element;
End;

Function SearchElement(Root: PWin2TreeElement; Handle: THandle): PWin2TreeElement; 
Var
  I : Integer;
Begin
  Result:=nil; If Root=nil Then Exit;

  If Handle=Root^.Handle Then
    Result:=Root
  Else If Root^.Childs<>nil Then For I:=0 To Root^.Childs.Count-1 Do Begin
    Result:=SearchElement(Root^.Childs.Items[I], Handle);
    If Result<>nil Then Exit;
  End;
End;

Procedure DestroyTree(Root: PWin2TreeElement); 
Begin
  If Root.Childs<>nil Then Begin
    While Root.Childs.Count>0 Do Begin
      DestroyTree(Root.Childs.Items[0]);
      Root.Childs.Delete(0);
    End;
    Root.Childs.Destroy;
  End;
  SafeFreeMem(Root, SizeOf(TWin2TreeElement));
End;

Function LoadTree_EnumWindowsProc(hwnd: HWND; lParam: LPARAM):bool stdcall; 
Var
  Owner   : THandle;
  Element : PWin2TreeElement;
Begin
  If MyIsWindowVisible(hwnd) Then Begin
    Owner   := GetWindow(hwnd, GW_OWNER);
    Element := SearchElement(PWin2TreeElement(lParam), Owner);
    If Element=nil Then Element:=CreateNewElement(PWin2TreeElement(lParam), Owner);
    If SearchElement(Element, hwnd)=nil Then CreateNewElement(Element, hwnd);
  End;
  Result := true;
End;

Function LoadTree:PWin2TreeElement;
Begin
  Result:=CreateNewElement(Nil, 0);
  EnumWindows(@LoadTree_EnumWindowsProc, LPARAM(Result));
End;

// -----------------------------------------------------------------------------
// Return text of window's caption
function  GetWindowTitle(W:THandle):ShortString; register;
Begin
  Result[0]:=Char(GetWindowText(W, @Result[1], SizeOf(Result)-1));
End;


function GetWindowModuleFileName(hwnd: HWND; buf:PChar; size:Integer):Integer; stdcall; external 'user32.dll' name 'GetWindowModuleFileName'

Procedure BuildMenu_Recursive(Root:PWin2TreeElement; PopupMenu:TMenuItem; OnClick:TNotifyEvent; ShowCordinates:Boolean; StickyWindows:TStickyWindows); 
Function CanBe(C:Char):Boolean; 
Var
  I, J: Integer;
Begin
  Result:=True;
  For I:=0 To PopupMenu.Count-1 Do Begin
    J:=Pos('&', PopupMenu.Items[I].Caption);
    If J>0 Then
      Result:=not (UpperCase(PopupMenu.Items[I].Caption[J+1])=UpperCase(C))
    Else Exit;
    If not Result Then Exit;
  End;
End;
Function AddHotkey(Const S:ShortString):ShortString; 
Var
  I : Integer;
Begin
  For I:=1 To Length(S) Do If CanBe(S[I]) Then Begin
    Result:=Copy(S, 1, I-1)+'&'+Copy(S, I, 255);
    Exit;
  End;
End;
Var
  I     : Integer;
  MenuI : TMenuItem;
  MenuI2: TMenuItem;
Begin
  MenuI         := TMenuItem.Create(PopupMenu);
  MenuI.Tag     := Root.Handle;
  If Assigned(StickyWindows) Then MenuI.Checked := StickyWindows.IsSticky(Root.Handle);
  PopupMenu.Add(MenuI);
  If Root.Childs=nil Then Begin
    MenuI.Caption := AddHotKey(Root.Title);
    MenuI.OnClick := OnClick;
  End Else Begin
    If Root.Childs.Count=1 Then
      MenuI.Caption := AddHotKey(Root.Title+' - '+PWin2TreeElement(Root.Childs.Items[0]).Title)
    Else Begin
      MenuI.Caption := AddHotKey(Root.Title);
      MenuI2:= TMenuItem.Create(MenuI);
      MenuI2.OnClick := OnClick;
      MenuI2.Tag     := -1;
      MenuI2.Caption := AddHotKey('This app');
      MenuI2.Enabled := not ShowCordinates;
      MenuI.Add(MenuI2);
      MenuI2:= TMenuItem.Create(MenuI);
      MenuI2.Caption := '-';
      MenuI.Add(MenuI2);

      For I:=0 To Root.Childs.Count-1 Do
        BuildMenu_Recursive(Root.Childs.Items[I], MenuI, OnClick, ShowCordinates, StickyWindows);
    End;
  End;
End;

Procedure BuildMenu(Root:PWin2TreeElement; PopupMenu:TMenuItem; OnClick:TNotifyEvent; ShowCordinates:Boolean; StickyWindows:TStickyWindows);
Var
  I     : Integer;
  W2T   : PWin2TreeElement;
Begin
  SortTree(Root);
  If Root.Childs<>nil Then
    For I:=0 To Root.Childs.Count-1 Do Begin
      W2T := PWin2TreeElement(Root.Childs.Items[I]);
      If (W2T.Title<>'')and
         (W2T.Title<>'Program Manager')and
         (W2T.Title<>'Semik''s desktop')and
         ((W2T.Rect.Left<>W2T.Rect.Right)or(W2T.Rect.Top<>W2T.Rect.Bottom)or
          (W2T.Childs<>nil)) Then
        BuildMenu_Recursive(Root.Childs.Items[I], PopupMenu, OnClick, ShowCordinates, StickyWindows);
    End;
End;

Procedure SortTree(Root:PWin2TreeElement);
Var
  I      : Integer;
  P      : Pointer;
  Sorted : Boolean;
Begin
  If Root.Childs<>nil Then If Root.Childs.Count>1 Then Begin
    // sort this level
    Repeat
      Sorted:=True;
      For I:=0 To Root.Childs.Count-2 Do
        If UpperCase(PWin2TreeElement(Root.Childs.Items[I]).Title)>UpperCase(PWin2TreeElement(Root.Childs.Items[I+1]).Title) Then Begin
          P:=Root.Childs.Items[I];
          Root.Childs.Items[I]:=Root.Childs.Items[I+1];
          Root.Childs.Items[I+1]:=P;
          Sorted:=False;
        End;
    Until Sorted;
    // sort lover level
    For I:=0 To Root.Childs.Count-1 Do SortTree(Root.Childs.Items[I]);
  End;
End;

Var
  CountWindowsByTitle_Counter : Integer;

Function CountWindowsByTitle_EnumWindowsProc(hwnd: HWND; lParam: LPARAM):bool stdcall;
Var
  Title   : Array[0..1024] of Char;
Begin
  GetWindowText(hwnd, @Title, SizeOf(Title));
  If StrComp(@Title, PChar(lParam))=0 Then Inc(CountWindowsByTitle_Counter);
  Result := true;
End;

Function CountWindowsByTitle(Const Caption : String):Integer; 
Var
  Title : Array[0..1024] of Char;
Begin
  CountWindowsByTitle_Counter:=0;
  Title[0]:=#0;
  StrPCopy(@Title[0], Caption);
  EnumWindows(@CountWindowsByTitle_EnumWindowsProc, Integer(@Title));
  Result:=CountWindowsByTitle_Counter;
End;

Const
  PSAPI_LIB          : THandle = 0;
  EnumProcessModules : Function (hProcess   : THANDLE; // handle to the process
                             var lphModule  : HMODULE; // array to receive the module handles
                                 cb         : DWORD;   // size of the array
                                 lpcbNeeded : LPDWORD  // receives the number of bytes returned);
                                ):Boolean far cdecl = nil;
  GetModuleFileNameEx: Function (hProcess   : THANDLE; // handle to the process
                                 hModule    : HMODULE; // handle to module to find filename for
                                 lpFilename : PChar;   // pointer to buffer to receive module path
                                 nSize      : DWORD    // size of buffer, in characters
                                ):DWORD far cdecl = nil;

{Var
  Buf           : Array[0..MAX_PATH] of Char;}

function GetWindowClass(W:THandle):String; register; 
{Var
  ProcessID, cb : DWORD;
  hProcess      : THandle;
  _Module       : HMODULE;
  BufWritten    : Integer;
  I             : Integer;}
Begin
{  SetLastError(0);
  If PSAPI_LIB<>0 Then Begin
    GetWindowThreadProcessId(W, @ProcessID);
    hProcess:=OpenProcess(PROCESS_QUERY_INFORMATION or PROCESS_VM_READ, False, ProcessID);
      I:=GetLastError;
    If EnumProcessModules(hProcess, _Module, SizeOf(_Module), @cb) Then Begin
      I:=GetLastError;
      FillChar(Buf, SizeOf(Buf), 0);
      BufWritten:=GetModuleFileNameEx(hProcess, _Module, @Buf[0], SizeOf(Buf));
      I:=GetLastError;
      Result[0]:=Char(BufWritten);
      For I:=0 To BufWritten-1 Do Result[I+1]:=Buf[I];
    End;
    //CloseHandle(hProcess);
  End Else }
  Result[0] := Char(GetClassName(W, @Result[1], SizeOf(Result)));
End;

initialization
  PSAPI_LIB := LoadLibrary('psapi.dll');
  If PSAPI_LIB<>0 Then Begin
    @EnumProcessModules := GetProcAddress(PSAPI_LIB, 'EnumProcessModules');
    @GetModuleFileNameEx:= GetProcAddress(PSAPI_LIB, 'GetModuleFileNameExA');
  End;
finalization
  If PSAPI_LIB<>0 Then FreeLibrary(PSAPI_LIB);
end.
