{*******************************************************}
{                                                       }
{         Delphi VCL Extensions (RX)                    }
{                                                       }
{         Copyright (c) 1997 Master-Bank                }
{                                                       }
{*******************************************************}

unit GIFCtrl;

interface

{$I RX.INC}

uses Messages, {$IFDEF WIN32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF}
  SysUtils, Classes, Graphics, Controls, Forms, StdCtrls, Menus, ExtCtrls,
  Animate, RxGIF, RxTimer;

type

{ TRxGIFAnimator }

  TRxGIFAnimator = class(TRxImageControl)
  private
    FAnimate: Boolean;
    FAutoSize: Boolean;
    FImage: TGIFImage;
    FTimer: TRxTimer;
    FFrameIndex: Integer;
    FStretch: Boolean;
    FLoop: Boolean;
    FCenter: Boolean;
    FTransparent: Boolean;
    FTimerRepaint: Boolean;
    FOnStart: TNotifyEvent;
    FOnStop: TNotifyEvent;
    FOnChange: TNotifyEvent;
    FOnFrameChanged: TNotifyEvent;
    function GetFrameBitmap(Index: Integer; var TransColor: TColor): TBitmap;
    function GetDelayTime(Index: Integer): Word;
    procedure AdjustBounds;
    procedure SetAutoSize(Value: Boolean);
    procedure SetAnimate(Value: Boolean);
    procedure SetCenter(Value: Boolean);
    procedure SetImage(Value: TGIFImage);
    procedure SetFrameIndex(Value: Integer);
    procedure SetStretch(Value: Boolean);
    procedure SetTransparent(Value: Boolean);
    procedure ImageChanged(Sender: TObject);
    procedure TimerExpired(Sender: TObject);
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
  protected
    function GetPalette: HPALETTE; override;
    procedure Paint; override;
    procedure DoPaintImage; override;
    procedure Change; dynamic;
    procedure FrameChanged; dynamic;
    procedure Start; dynamic;
    procedure Stop; dynamic;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Animate: Boolean read FAnimate write SetAnimate default False;
    property AutoSize: Boolean read FAutoSize write SetAutoSize default True;
    property Center: Boolean read FCenter write SetCenter default False;
    property FrameIndex: Integer read FFrameIndex write SetFrameIndex default 0;
    property Image: TGIFImage read FImage write SetImage;
    property Loop: Boolean read FLoop write FLoop default True;
    property Stretch: Boolean read FStretch write SetStretch default False;
    property Transparent: Boolean read FTransparent write SetTransparent default True;
{$IFDEF RX_D4}
    property Anchors;
    property Constraints;
    property DragKind;
{$ENDIF}
    property Align;
    property Cursor;
    property DragCursor;
    property DragMode;
    property Enabled;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnFrameChanged: TNotifyEvent read FOnFrameChanged write FOnFrameChanged;
    property OnStart: TNotifyEvent read FOnStart write FOnStart;
    property OnStop: TNotifyEvent read FOnStop write FOnStop;
    property OnClick;
    property OnDblClick;
    property OnDragOver;
    property OnDragDrop;
    property OnEndDrag;
    property OnMouseMove;
    property OnMouseDown;
    property OnMouseUp;
{$IFDEF WIN32}
    property OnStartDrag;
{$ENDIF}
{$IFDEF RX_D4}
    property OnEndDock;
    property OnStartDock;
{$ENDIF}
  end;

implementation

uses VCLUtils, MaxMin, RxGraph;

{ TRxGIFAnimator }

constructor TRxGIFAnimator.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FTimer := TRxTimer.Create(Self);
  FImage := TGIFImage.Create;
  FGraphic := FImage;
  FImage.OnChange := ImageChanged;
  FAutoSize := True;
  FLoop := True;
  FTransparent := True;
end;

destructor TRxGIFAnimator.Destroy;
begin
  FOnStart := nil;
  FOnStop := nil;
  FOnChange := nil;
  FOnFrameChanged := nil;
  Animate := False;
  FImage.OnChange := nil;
  FImage.Free;
  inherited Destroy;
end;

procedure TRxGIFAnimator.AdjustBounds;
begin
  if not (csReading in ComponentState) then begin
    if FAutoSize and not FImage.Empty then
      SetBounds(Left, Top, FImage.ScreenWidth, FImage.ScreenHeight);
  end;
end;

function TRxGIFAnimator.GetDelayTime(Index: Integer): Word;
begin
  if (FFrameIndex >= 0) and (FFrameIndex < FImage.Count) and
    (FImage.Count > 1) then
    Result := Max(FImage.Frames[FFrameIndex].AnimateInterval, 1)
  else Result := 0;
end;

function TRxGIFAnimator.GetFrameBitmap(Index: Integer;
  var TransColor: TColor): TBitmap;
var
  I, Last, First: Integer;
  SavePal: HPalette;
begin
  Result := TBitmap.Create;
  try
    with Result do begin
      Width := FImage.ScreenWidth;
      Height := FImage.ScreenHeight;
      SavePal := 0;
      if FImage.Palette <> 0 then begin
        SavePal := SelectPalette(Canvas.Handle, FImage.Palette, False);
        RealizePalette(Canvas.Handle);
      end;
      if (FImage.Frames[FImage.FrameIndex].TransparentColor <> clNone) then
      begin
        TransColor := GetNearestColor(Canvas.Handle,
          ColorToRGB(FImage.Frames[FImage.FrameIndex].TransparentColor));
        Canvas.Brush.Color := PaletteColor(TransColor);
      end
      else if (FImage.BackgroundColor <> clNone) and FImage.Transparent then
        Canvas.Brush.Color := PaletteColor(FImage.BackgroundColor)
      else Canvas.Brush.Color := PaletteColor(clWindow);
      Canvas.FillRect(Bounds(0, 0, Width, Height));
      Last := Min(Index, FImage.Count - 1);
      First := Max(0, Last - 1);
      while First > 0 do begin
        if (FImage.Frames[First].DisposalMethod = dmRestoreBackground) and
          (FImage.ScreenWidth = FImage.Frames[First].Width) and
          (FImage.ScreenHeight = FImage.Frames[First].Height) then
          Break;
        Dec(First);
      end;
      for I := First to Last - 1 do begin
        with FImage.Frames[I] do
          case DisposalMethod of
            dmUndefined, dmLeave:
              Draw(Canvas, Bounds(Origin.X, Origin.Y, Width, Height), True);
            dmRestoreBackground:
              if I > First then
                Canvas.FillRect(Bounds(Origin.X, Origin.Y, Width, Height));
            dmRestorePrevious:
              begin { do nothing } end;
          end;
      end;
      with FImage.Frames[Last] do
        Draw(Canvas, Bounds(Origin.X, Origin.Y, Width, Height), True);
{$IFDEF RX_D3}
      if (TransColor <> clNone) and FTransparent then begin
        TransparentColor := PaletteColor(TransColor);
        Transparent := True;
      end;
{$ENDIF}
      if FImage.Palette <> 0 then
        SelectPalette(Canvas.Handle, SavePal, False);
    end;
  except
    Result.Free;
    raise;
  end;
end;

function TRxGIFAnimator.GetPalette: HPALETTE;
begin
  Result := 0;
  if not FImage.Empty then Result := FImage.Palette;
end;

procedure TRxGIFAnimator.ImageChanged(Sender: TObject);
begin
  AdjustBounds;
  FFrameIndex := FImage.FrameIndex;
  if (FFrameIndex >= 0) and (FImage.Count > 0) then
    FTimer.Interval := GetDelayTime(FFrameIndex);
  Change;
  PictureChanged;
end;

procedure TRxGIFAnimator.SetImage(Value: TGIFImage);
begin
  FImage.Assign(Value);
end;

procedure TRxGIFAnimator.SetCenter(Value: Boolean);
begin
  if Value <> FCenter then begin
    FCenter := Value;
    PictureChanged;
    if Animate then Repaint;
  end;
end;

procedure TRxGIFAnimator.SetStretch(Value: Boolean);
begin
  if Value <> FStretch then begin
    FStretch := Value;
    PictureChanged;
    if Animate then Repaint;
  end;
end;

procedure TRxGIFAnimator.SetTransparent(Value: Boolean);
begin
  if Value <> FTransparent then begin
    FTransparent := Value;
    PictureChanged;
    if Animate then Repaint;
  end;
end;

procedure TRxGIFAnimator.SetFrameIndex(Value: Integer);
begin
  if Value <> FFrameIndex then begin
    if (Value < FImage.Count) and (Value >= 0) then begin
      FFrameIndex := Value;
      if (FFrameIndex >= 0) and (FImage.Count > 0) then
        FTimer.Interval := GetDelayTime(FFrameIndex);
      FrameChanged;
      PictureChanged;
    end;
  end;
end;

procedure TRxGIFAnimator.DoPaintImage;
var
  TmpImage, Frame: TBitmap;
  Dest: TRect;
  TransColor: TColor;
begin
  TmpImage := TBitmap.Create;
  try
    with TmpImage do begin
      Width := ClientWidth;
      Height := ClientHeight;
      Canvas.Brush.Color := Self.Color;
      Canvas.FillRect(Bounds(0, 0, Width, Height));
      { copy image from parent and back-level controls }
      if FImage.Transparent or FImage.Empty then
        CopyParentImage(Self, Canvas);
      if not FImage.Empty and (FImage.ScreenWidth > 0) and
        (FImage.ScreenHeight> 0) then
      begin
        TransColor := clNone;
        Frame := GetFrameBitmap(FrameIndex, TransColor);
        try
          if FStretch then
            Dest := Self.ClientRect
          else if FCenter then
            Dest := Bounds((Self.ClientWidth - Frame.Width) div 2,
              (Self.ClientHeight - Frame.Height) div 2,
              Frame.Width, Frame.Height)
          else
            Dest := Rect(0, 0, Frame.Width, Frame.Height);
          if (TransColor = clNone) or not FTransparent then
            Canvas.StretchDraw(Dest, Frame)
          else begin
            StretchBitmapRectTransparent(Canvas, Dest.Left, Dest.Top,
              WidthOf(Dest), HeightOf(Dest), Bounds(0, 0, Frame.Width,
              Frame.Height), Frame, TransColor);
          end;
        finally
          Frame.Free;
        end;
      end;
    end;
    Canvas.Draw(ClientRect.Left, ClientRect.Top, TmpImage);
  finally
    TmpImage.Free;
  end;
end;

procedure TRxGIFAnimator.Paint;
begin
  PaintImage;
  if (FImage.Transparent or FImage.Empty) then
    PaintDesignRect;
end;

procedure TRxGIFAnimator.TimerExpired(Sender: TObject);
var
  F: TCustomForm;
begin
  if Visible and (FImage.Count > 1) then begin
    if FFrameIndex < FImage.Count - 1 then Inc(FFrameIndex)
    else FFrameIndex := 0;
    FTimerRepaint := True;
    try
      FrameChanged;
      Repaint;
    finally
      FTimerRepaint := False;
      if (FFrameIndex >= 0) and (FFrameIndex < FImage.Count) then
        FTimer.Interval := GetDelayTime(FFrameIndex);
    end;
    if not FLoop and (FFrameIndex = 0) then begin
      SetAnimate(False);
      if (csDesigning in ComponentState) then begin
        F := GetParentForm(Self);
        if (F <> nil) and (F.Designer <> nil) then
          F.Designer.Modified;
      end;
    end;
  end;
end;

procedure TRxGIFAnimator.Change;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TRxGIFAnimator.FrameChanged;
begin
  if Assigned(FOnFrameChanged) then FOnFrameChanged(Self);
end;

procedure TRxGIFAnimator.Stop;
begin
  if Assigned(FOnStop) then FOnStop(Self);
end;

procedure TRxGIFAnimator.Start;
begin
  if Assigned(FOnStart) then FOnStart(Self);
end;

procedure TRxGIFAnimator.SetAutoSize(Value: Boolean);
begin
  if Value <> FAutoSize then begin
    FAutoSize := Value;
    AdjustBounds;
    PictureChanged;
  end;
end;

procedure TRxGIFAnimator.SetAnimate(Value: Boolean);
begin
  if FAnimate <> Value then begin
    if Value then begin
      FTimer.OnTimer := TimerExpired;
      FTimer.Enabled := True;
      FAnimate := FTimer.Enabled;
      Start;
    end
    else begin
      FTimer.Enabled := False;
      FTimer.OnTimer := nil;
      FAnimate := False;
      Stop;
      PictureChanged;
    end;
  end;
end;

procedure TRxGIFAnimator.WMSize(var Message: TWMSize);
begin
  inherited;
  AdjustBounds;
end;

end.